/**
 * XSplit Express Project File (.xveproj) Reader for Cue Points
 * 
 * XSplit Express Editor stores markers/cue points in .xveproj project files,
 * not in the MP4 files themselves. This module attempts to read cue points
 * from the project file if it exists.
 */

const { readFileSync, existsSync } = require('fs');
const { join, dirname, basename } = require('path');
const logger = require('../logger');

/**
 * Find potential .xveproj project file for a given MP4 file
 * @param {string} mp4Path - Path to the MP4 file
 * @returns {string|null} Path to the project file, or null if not found
 */
function findProjectFile(mp4Path) {
  const dir = dirname(mp4Path);
  const baseName = basename(mp4Path, '.mp4');
  
  // Common project file naming patterns:
  // - Same name as MP4: video.mp4 -> video.xveproj
  // - "Untitled.xveproj" in the same directory
  // - Any .xveproj file in the same directory
  
  const candidates = [
    join(dir, `${baseName}.xveproj`),
    join(dir, 'Untitled.xveproj'),
    // Could also search for all .xveproj files, but let's start with these
  ];
  
  for (const candidate of candidates) {
    if (existsSync(candidate)) {
      logger.log(`[CuePoints] Found project file: ${candidate}`);
      return candidate;
    }
  }
  
  // Try to find any .xveproj file in the directory
  try {
    const fs = require('fs');
    const files = fs.readdirSync(dir);
    const xveprojFiles = files.filter(f => f.toLowerCase().endsWith('.xveproj'));
    if (xveprojFiles.length > 0) {
      const projectPath = join(dir, xveprojFiles[0]);
      logger.log(`[CuePoints] Found project file in directory: ${projectPath}`);
      return projectPath;
    }
  } catch (err) {
    logger.warn(`[CuePoints] Error searching for project files: ${err.message}`);
  }
  
  return null;
}

/**
 * Read cue points from XSplit Express project file (.xveproj)
 * 
 * Note: The .xveproj format is likely XML or binary. This is a first attempt
 * to read it. We'll need to examine actual project files to understand the format.
 * 
 * @param {string} mp4Path - Path to the MP4 file (used to find the project file)
 * @returns {Promise<Array>} Array of cue point objects with {startTime, endTime, title}
 */
async function readCuePointsFromProjectFile(mp4Path) {
  const projectPath = findProjectFile(mp4Path);
  
  if (!projectPath) {
    logger.log('[CuePoints] No .xveproj project file found for:', mp4Path);
    return [];
  }
  
  try {
    logger.log('[CuePoints] Reading project file:', projectPath);
    const fileContent = readFileSync(projectPath);
    
    // Try to determine if it's text (XML/JSON) or binary
    let isText = false;
    let textContent = null;
    
    try {
      textContent = fileContent.toString('utf8');
      // Check if it looks like text (mostly printable ASCII)
      const printableRatio = Array.from(textContent.slice(0, 1000))
        .filter(c => c.charCodeAt(0) >= 32 && c.charCodeAt(0) < 127).length / Math.min(1000, textContent.length);
      isText = printableRatio > 0.7;
    } catch (e) {
      // Not valid UTF-8, probably binary
      isText = false;
    }
    
    if (isText && textContent) {
      logger.log('[CuePoints] Project file appears to be text format');
      logger.log('[CuePoints] First 500 chars:', textContent.substring(0, 500));
      
      // Try to parse as XML
      if (textContent.trim().startsWith('<?xml') || textContent.trim().startsWith('<')) {
        logger.log('[CuePoints] Detected XML format');
        // TODO: Parse XML to find marker/cue point elements
        // We'll need to examine actual .xveproj files to understand the structure
        logger.warn('[CuePoints] XML parsing not yet implemented - need to examine .xveproj file structure');
        return [];
      }
      
      // Try to parse as JSON
      try {
        const json = JSON.parse(textContent);
        logger.log('[CuePoints] Detected JSON format');
        logger.log('[CuePoints] JSON keys:', Object.keys(json));
        // TODO: Extract cue points from JSON structure
        logger.warn('[CuePoints] JSON parsing not yet implemented - need to examine .xveproj file structure');
        return [];
      } catch (e) {
        logger.log('[CuePoints] Not JSON format');
      }
    } else {
      logger.log('[CuePoints] Project file appears to be binary format');
      logger.log('[CuePoints] File size:', fileContent.length, 'bytes');
      logger.log('[CuePoints] First 100 bytes (hex):', Array.from(fileContent.slice(0, 100))
        .map(b => b.toString(16).padStart(2, '0')).join(' '));
      // TODO: Parse binary format
      logger.warn('[CuePoints] Binary format parsing not yet implemented - need to examine .xveproj file structure');
    }
    
    logger.warn('[CuePoints] Project file format not yet supported - need to reverse engineer .xveproj structure');
    return [];
    
  } catch (error) {
    logger.error('[CuePoints] Error reading project file:', error.message);
    return [];
  }
}

module.exports = {
  findProjectFile,
  readCuePointsFromProjectFile
};
